<#
 disable stale accounts

 Author: Michael Zier

 Version: 1.2.0

 Changelog:
  2020-05-11: initial version
  2020-06-26: send mail of last months disabled accounts on every 1st
  2020-09-18: moved config to xml
#>
#requires -version 3.0
Param(
    [Parameter(Mandatory=$false, HelpMessage='Config XML')][string] $Xml = '.\Disabe-StaleAccounts.xml'
)


try {
    $Cfg = ([xml](Get-Content -Path $Xml)).mz
}
catch {
    Write-Host 'Error reading config!' -ForegroundColor Red
    Exit 1603
}
if ($Cfg.Mail.From.Email -notmatch  "^\w+([-+.']\w+)*@\w+([-.]\w+)*\.\w+([-.]\w+)*$") {
    Write-Host "Invalid mail address set in $Xml" -ForegroundColor Red
    Exit 1603
}

#Region Logging
# logging
if ($Cfg.Log.UseUtc -eq 'True') {
    $boolUseUtc = $true
    $strDate = (Get-Date).ToUniversalTime().ToString('yyyy-MM-dd')
    $strDateMonth = (Get-Date).ToUniversalTime().ToString('yyyy-MM')
}
else {
    $boolUseUtc = $false
    $strDate = Get-Date -Format 'yyyy-MM-dd'
    $strDateMonth = Get-Date -Format 'yyyy-MM'
}
$strLogDir           = "$PSScriptRoot\$($Cfg.Log.Directory)"
$strLogFile          = "$strLogDir\$($Cfg.Log.File)"     -replace '%DATE%', $strDate
$strELogFile         = "$strLogDir\$($Cfg.Log.Error)"    -replace '%DATE%', $strDate
$strDisabledAccounts = "$strLogDir\$($Cfg.Log.Disabled)" -replace '%MONTH%', $strDateMonth

function Write-Log ($strText) {
    if ($boolUseUtc) {
        $strDateTime = (Get-Date).ToUniversalTime().ToString('yyyy/MM/dd - HH:mm:ss') # UTC
    } else {
        $strDateTime = Get-Date -Format 'yyyy/MM/dd - HH:mm:ss' # client timezone
    }
    Add-Content -Path $strLogFile -Value "[$strDateTime] $strText"
        switch -Wildcard ($strText) {
            '`[INFO`]*'    { Write-Host $strText; break }
            '`[WARNING`]*' { Write-Host $strText -ForegroundColor Yellow ; break }
            '`[ERROR`]*'   {
                Write-Host $strText -ForegroundColor Red -BackgroundColor Black
                Add-Content -Path $strELogFile -Value "[$strDateTime] $strText"
                break
            }
            '`[SUCCESS`]*' { Write-Host $strText -ForegroundColor Green; break }
            '`[FATAL`]*'   { Write-Host $strText -BackgroundColor Red; break }
            default        { Write-Host $strText }
        }
}
#EndRegion

# set default -Server parameter for AD cmdlets
$PSDefaultParameterValues['*-AD*:Server'] = $Cfg.ActiveDirectory.DomainController

try {
    Write-Log "[INFO] Getting inactive users"
    # UserAccountControl = 2080 are trusts
    $objUsers = Get-ADUser -Filter {Enabled -eq $true -and UserAccountControl -ne 2080} -Properties Created,LastLogonDate,userAccountControl -SearchBase $Cfg.ActiveDirectory.SearchBase | Sort-Object -Property SamAccountName
}
catch {
    Write-Log "[ERROR] Unable to get userlist: $_"
    exit 1603
}

$objInactiveDate = (Get-Date).AddDays(-$Cfg.General.DisableAfter)

foreach ($objUser in $objUsers) {
    # password never expires
    if ($Cfg.Gerneral.IgnorePwdNeverExpires -eq 'True' -and $objUser.userAccountControl -eq 66048) {
        Write-Log "[INFO] Skipped:  $($objUser.SamAccountName) (password never expires)"
        continue
    }
    # excluded accounts
    if ($Cfg.Exclusion.SAM -contains $objUser.SamAccountName) {
        Write-Log "[INFO] Skipped:  $($objUser.SamAccountName) (member of exclusionlist)"
        continue
    }
    if ($objUser.LastLogonDate -gt $objInactiveDate) {
        if ($Cfg.Log.LogActiveUser -eq 'True') {
            Write-Log "[INFO] Active:   $($objUser.SamAccountName) (Lastlogon: $($objUser.LastLogonDate))"
        }
        continue
    }
    if ($objUser.Created -gt $objInactiveDate) {
        if ($Cfg.Log.LogNewUser -eq 'True') {
            Write-Log "[INFO] New:      $($objUser.SamAccountName) (Created:   $($objUser.Created))"
        }
        continue
    }

    try {
        if (-not $objUser.LastLogonDate) {
            Write-Log "[INFO]  Disabling $($objUser.SamAccountName) (Created: $($objUser.Created))"
        }
        else {
            Write-Log "[INFO]  Disabling $($objUser.SamAccountName) (Lastlogon: $($objUser.LastLogonDate))"
        }
        $objUser | Disable-ADAccount
        $objUser | Select SamAccountName, DistinguishedName, LastLogonDate, Created | Export-Csv -Path $strDisabledAccounts -Append -NoTypeInformation
    }
    catch {
        Write-Log "[ERROR] $_"
    }
}

#Region send mail
if ((Get-Date).Day -eq 1) {
    $strDateLastMonth = (Get-Date).ToUniversalTime().AddMonths(-1).ToString('yyyy-MM')
    $strLastCsv = "$PSScriptRoot\Logs\$($Cfg.Log.Disabled)" -replace '%MONTH%', $strDateLastMonth
    $intCount = (Import-Csv -Path $strLastCsv).count

    $strSubject = "$intCount Disabled Inactive Accounts $((Get-Date).ToUniversalTime().AddMonths(-1).ToString('yyyy/MM'))"
    $strMsg = "$intCount inactive Active Directory accounts have been disabled last month"
    # send mail
    try {
        Write-Log "[INFO] Sending mail: $strSubject"
        Send-MailMessage -From "$($Cfg.Mail.From.Name) <$($Cfg.Mail.From.Email)>" -To $Cfg.Mail.To -Subject $strSubject -Body $strMsg -Attachments $strLastCsv -SmtpServer $Cfg.Mail.Smtp -ErrorAction Stop
    }
    catch {
        Write-Log "[ERROR] $_"
    }
}
#EndRegion

Write-Log '[SUCCESS] Done!'