<#
 Send password expires reminder mails

 Author: Michael Zier

 Version: 1.1.0

 Changelog:
  2019-12-18: initial version
  2020-03-25: send mail also to managedBy
  2020-05-04: save accounts with PasswordNeverExpires to extra file
  2020-05-12: save accounts with empty pwdLastset to extra file
  2020-09-11: moved config to XML
#>
#requires -version 3.0
Param(
    [Parameter(Mandatory=$false, HelpMessage='Config XML')][string] $Xml = '.\Send-PwdExpReminder.xml'
)


try {
    $Cfg = ([xml](Get-Content -Path $Xml)).mz
}
catch {
    Write-Host 'Error reading config!' -ForegroundColor Red
    Exit 1603
}
if ($Cfg.Mail.From.Email -notmatch  "^\w+([-+.']\w+)*@\w+([-.]\w+)*\.\w+([-.]\w+)*$") {
    Write-Host "Invalid mailfrom address set in $Xml" -ForegroundColor Red
    Exit 1603
}

#Region logging
if ($Cfg.General.Log.UseUtc -eq 'True') {
    $boolUseUtc = $true
    $strDate = (Get-Date).ToUniversalTime().ToString('yyyy-MM-dd')
}
else {
    $boolUseUtc = $false
    $strDate = Get-Date -Format 'yyyy-MM-dd'
}
$strLogDir          = "$PSScriptRoot\$($Cfg.General.Log.Directory)"
$strLogFile         = "$strLogDir\$($Cfg.General.Log.File)"         -replace '%DATE%', $strDate
$strELogFile        = "$strLogDir\$($Cfg.General.Log.Error)"        -replace '%DATE%', $strDate
$strPwdNeverExpires = "$strLogDir\$($Cfg.General.Log.NeverExpires)" -replace '%DATE%', $strDate
$strPwdNeverSet     = "$strLogDir\$($Cfg.General.Log.NeverSet)"     -replace '%DATE%', $strDate

function Write-Log ($strText) {
    if ($boolUseUtc) {
        $strDateTime = (Get-Date).ToUniversalTime().ToString('yyyy/MM/dd - HH:mm:ss') # UTC
    } else {
        $strDateTime = Get-Date -Format 'yyyy/MM/dd - HH:mm:ss' # client timezone
    }
    Add-Content -Path $strLogFile -Value "[$strDateTime] $strText"
        switch -Wildcard ($strText) {
            '`[INFO`]*'    { Write-Host $strText; break }
            '`[WARNING`]*' { Write-Host $strText -ForegroundColor Yellow ; break }
            '`[ERROR`]*'   {
                Write-Host $strText -ForegroundColor Red -BackgroundColor Black
                Add-Content -Path $strELogFile -Value "[$strDateTime] $strText"
                break
            }
            '`[SUCCESS`]*' { Write-Host $strText -ForegroundColor Green; break }
            '`[FATAL`]*'   { Write-Host $strText -BackgroundColor Red; break }
            default        { Write-Host $strText }
        }
}
#EndRegion

#Region send mail function
function Send-Reminder {
    param (
        [Parameter(Mandatory=$true)][string] $SamAccountName,
        [Parameter(Mandatory=$true)][string] $MailTo,
        [Parameter(Mandatory=$true)][string] $PwdExpireDate
    )

    if ($MailTo -match "^\w+([-+.']\w+)*@\w+([-.]\w+)*\.\w+([-.]\w+)*$") {
        # replace #DOMAIN#, #ACCOUNT#, #EXPIREDATE# in mail template
        $strMailMsg = $strMailTpl -replace '#DOMAIN#', $Cfg.ActiveDirectory.Domain
        $strMailMsg = $strMailMsg -replace '#ACCOUNT#', $SamAccountName
        $strMailMsg = $strMailMsg -replace '#EXPIREDATE#', $PwdExpireDate
        $strMailMsg = $strMailMsg | Out-String # we need this to make new lines work
        Write-Log "[INFO]  ${SamAccountName}: Sending reminder to <$($MailTo)>"
        # send reminder mail
        try {
            Send-MailMessage -From "$($Cfg.Mail.From.Name) <$($Cfg.Mail.From.Email)>" -To $MailTo -Subject 'Password Expiry Reminder' -Body $strMailMsg -Encoding ([System.Text.Encoding]::UTF8) -Priority High -SmtpServer $Cfg.Mail.Smtp -ErrorAction Stop
        }
        catch {
            Write-Log "[ERROR] ${SamAccountName}: Send-MailMessage: $_"
            return $false
        }
    }
    else {
        Write-Log "[ERROR] ${SamAccountName}: Invalid mail attribute <$MailTo>"
        return $false
    }
    return $true
}
#EndRegion

# set default -Server parameter for AD cmdlets
$PSDefaultParameterValues['*-AD*:Server'] = $Cfg.ActiveDirectory.DomainController

#Region get users
try {
    Write-Log "[INFO] Getting users"
    # UserAccountControl = 2080 are trusts
    $objUsers = Get-ADUser -Filter {Enabled -eq $true -and UserAccountControl -ne 2080} -Properties PasswordNeverExpires,PasswordLastSet,mail,"msDS-UserPasswordExpiryTimeComputed",managedBy -SearchBase $Cfg.ActiveDirectory.SearchBase
}
catch {
    Write-Log "[ERROR] Unable to get userlist: $_"
}
#EndRegion

# read mail template
$strMailTpl = Get-Content $Cfg.Mail.Template

#Region walk through users
foreach ($objUser in $objUsers) {
    # skip users with PasswordNeverExpires set
    if ($objUser.PasswordNeverExpires -eq $true) {
        Write-Log "[INFO]  $($objUser.SamAccountName): PasswordNeverExpires set"
        $objUser | Select SamAccountName,PasswordLastSet,mail,DistinguishedName,managedBy | Export-Csv -Path $strPwdNeverExpires -Append -NoTypeInformation
        continue
    }
    # skip users with PasswordLastSet not set
    if (-not $objUser.PasswordLastSet) {
        Write-Log "[INFO]  $($objUser.SamAccountName): Password never changed"
        $objUser | Select SamAccountName,mail,DistinguishedName,managedBy | Export-Csv -Path $strPwdNeverSet -Append -NoTypeInformation
        continue
    }

    # calculate password expire date
    $objPwdExpireDate = [datetime]::FromFileTime($objUser."msDS-UserPasswordExpiryTimeComputed")
    # calculate days until password expires
    $objPwdExpiresIn = $objPwdExpireDate - (Get-Date)

    if ($objPwdExpiresIn.Days -ge 0) {
        Write-Log "[INFO]  $($objUser.SamAccountName): Password expires in $($objPwdExpiresIn.Days) days (Password last set: $($objUser.PasswordLastSet))"
    }
    else {
        Write-Log "[INFO]  $($objUser.SamAccountName): Password expired since $($objPwdExpiresIn.Days*-1) days (Password last set: $($objUser.PasswordLastSet))"
    }

    # check if mail should be send and send it (or not)
    foreach ($RemindBefore in $Cfg.General.RemindBefore.Days) {
        if ($objPwdExpiresIn.Days -eq $RemindBefore) {
            # send mail to user
            Send-Reminder $objUser.SamAccountName $objUser.mail $objPwdExpireDate
            # check for managedby
            if ($objUser.managedBy -ne $null) {
                $objManager = $objUsers | Where { $_.DistinguishedName -eq $objUser.managedBy }
                Write-Log "[INFO]  $($objUser.SamAccountName): ManagedBy $($objManager.SamAccountName)"
                # send mail to manager
                Send-Reminder $objUser.SamAccountName $objManager.mail $objPwdExpireDate
            }
        }
    }
}
#EndRegion

Write-Log '[SUCCESS] Done!'