﻿<#

 Script to remove the unwanted Apps which are coming with Windows 10

 Author: Michael Zier <michael.zier@mahe.at>

 Version: 1.5.0

 Changelog:
  2015-10-03: initial Version
  2016-09-15: 1.0.1 added new Appx
  2017-02-09: 1.1.0
              added support for provisioned packages
              extended remove list
              different logging path when running in Task Sequence and System
  2017-02-13: 1.2.0 support for offline mode
  2017-07-10: 1.2.1 added 3DViewer and MSPaint to the list
  2017.07-11: 1.3.0 now supports Windows packages
  2018-01-18: 1.3.1 1709
  2018-01-31: 1.3.2 added support for usercontext
  2018-05-11: 1.3.3 write RegKey to avoid returning of the Appx (see: https://docs.microsoft.com/en-us/windows/application-management/remove-provisioned-apps-during-update)
  2018-05-25: 1.3.4 1803
  2018-06-16: 1.4.0 switched to XML
  2018-08-03: 1.5.0 setting Policy RegKey for disabling consumer features
#>
#requires -version 3.0
Param(
    [Parameter(Mandatory=$false, HelpMessage='Config XML')][string] $Xml = '.\Remove-UWP.xml',
    [Parameter(Mandatory=$false, HelpMessage='Do not hide PS Windows')][Switch] $NoHide
)

if (-not (Test-Path $Xml)) {
    Write-Host 'Config not found!' -ForegroundColor Red
    Exit 3
}
try {
    $Cfg = ([xml](Get-Content -Path $Xml)).mz
}
catch {
    Write-Host 'Error reading config!' -ForegroundColor Red
    Exit 1603
}

# logfile
$strLogFile = $Cfg.General.Log

# Appx to remove
$objUwpRemove = $Cfg.UWP.Item | Where Action -eq 'Remove' | Select Name

# Windows packages to remove
$objWpRemove = $Cfg.WP.Item | Where Action -eq 'Remove' | Select Name

$strAppxRegKey = 'HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Appx\AppxAllUserStore\Deprovisioned'

## functions
# logging
function Write-Log ($strText) {
    #$strDateTime = Get-Date -Format 'yyyy/MM/dd - HH:mm:ss' # client timezone
    $strDateTime = (Get-Date).ToUniversalTime().ToString('yyyy/MM/dd - HH:mm:ss') # UTC
    Add-Content -Path $strLogFile -Value "[$strDateTime] $strText"
    Write-Host $strText
}

try {
    # in TS
    $objTSEnv = New-Object -COMObject Microsoft.SMS.TSEnvironment
    # logfile
    $strLogFile = "$($objTsEnv.Value('_SMSTSLogPath'))\$strLogFile"
    Write-Log "Running in a Task Sequence"
    $strContext = 'TS'
} catch {
    #if ($Env:USERNAME -eq "$($Env:COMPUTERNAME)`$") {
    if (([System.Security.Principal.WindowsIdentity]::GetCurrent()).IsSystem) {
        # system context
        $strLogFile = "$($Env:SystemRoot)\Logs\$strLogFile"
        Write-Log 'Running in SYSTEM context'
        $strContext = 'SYSTEM'
    } else {
        # user context
        if (-not $NoHide) {
            # hide Powershell Window
            Add-Type -Name win -MemberDefinition '[DllImport("user32.dll")] public static extern bool ShowWindow(int handle, int state);' -Namespace native
            [native.win]::ShowWindow(([System.Diagnostics.Process]::GetCurrentProcess() | Get-Process).MainWindowHandle,0)
        }
        if ([bool](([System.Security.Principal.WindowsIdentity]::GetCurrent()).groups -match "S-1-5-32-544")) {
            $IsAdmin = $true
        }
        $strLogFile = "$($Env:LocalAppdata)\$strLogFile"
        Write-Log 'Running in USER context'
        $strContext = 'USER'
    }
}

if ($env:SYSTEMDRIVE -eq "X:") {
    $boolOffline = $true
    # Find Windows
    $drives = get-volume | ? {-not [String]::IsNullOrWhiteSpace($_.DriveLetter) } | ? {$_.DriveType -eq 'Fixed'} | ? {$_.DriveLetter -ne 'X'}
    $drives | ? { Test-Path "$($_.DriveLetter):\Windows\System32"} | % { $strOfflinePath = "$($_.DriveLetter):\" }
    Write-Log "Eligible offline drive found: $strOfflinePath"
} else {
    Write-Log "Running on full OS."
    $boolOffline = $false
}

# move move move!
if ($boolOffline) {
    Start-Process -FilePath REG.EXE -ArgumentList "LOAD HKLM\SWOFF ${strOfflinePath}Windows\System32\Config\SOFTWARE"
    while (-not (Test-Path 'Registry::HKEY_LOCAL_MACHINE\SWOFF')) { Start-Sleep -Seconds 3 } # wait for the hive to be loaded
    $strAppxRegKey = $strAppxRegKey -replace 'HKLM:\\SOFTWARE','HKLM:\SWOFF'

    if ($Cfg.General.DisableWindowsconsumerFeatures -eq 'True') {
        try {
            Write-Log "Disabling Consumer Features"
            New-Item -Path 'HKLM:\SWOFF\Policies\Microsoft\Windows\CloudContent' -Force -ErrorAction SilentlyContinue
            New-ItemProperty -Path 'HKLM:\SWOFF\Policies\Microsoft\Windows\CloudContent' -Name 'DisableWindowsconsumerFeatures' -Value 1 -PropertyType DWord -Force
            Write-Log "Success"
        } catch {
            Write-Log "Error: $_"
        }
    }
} else {
    if (($IsAdmin -or $strContext -ne 'USER') -and $Cfg.General.DisableWindowsconsumerFeatures -eq 'True') {
        try {
            Write-Log "Disabling Consumer Features"
            New-Item -Path 'HKLM:\Software\Policies\Microsoft\Windows\CloudContent' -Force -ErrorAction SilentlyContinue
            New-ItemProperty -Path 'HKLM:\Software\Policies\Microsoft\Windows\CloudContent' -Name 'DisableWindowsconsumerFeatures' -Value 1 -PropertyType DWord -Force
            Write-Log "Success"
        } catch {
            Write-Log "Error: $_"
        }
    }
}

foreach ($remove in $objUwpRemove) {
    if (!$boolOffline) {
        # Appx package
        $objRemove = Get-AppxPackage -Name $remove.Name
        if ($objRemove -ne $null) {
            Write-Log "Removing Appx package: $($remove.Name)"
            try {
                Remove-AppxPackage -Package $objRemove.PackageFullName
                Write-Log "Success"
                Start-Sleep -Seconds 3
            } catch {
                Write-Log "Error: $_"
            }
        } else {
            Write-Log "Appx package not found: $($remove.Name)"
        }
    }

    if ($IsAdmin -or $strContext -ne 'USER') {
        # Appx provisioned package
        if ($boolOffline) {
            $objProvRemove = Get-AppxProvisionedPackage -Path $strOfflinePath | Where { $_.DisplayName -eq $remove.Name }
        } else {
            $objProvRemove = Get-AppxProvisionedPackage -Online | Where { $_.DisplayName -eq $remove.Name }
        }
        if ($objProvRemove -ne $null) {
            Write-Log "Removing Appx provisioned package: $($remove.Name)"
            try {
                if ($boolOffline) {
                    Remove-AppxProvisionedPackage -Path $strOfflinePath -PackageName $objProvRemove.Packagename
                } else {
                    Remove-AppxProvisionedPackage -online -PackageName $objProvRemove.Packagename
                }
                New-Item -Path $strAppxRegKey -Name $objProRemove.Packagename -Force
                Write-Log "Success"
                Start-Sleep -Seconds 3
            } catch {
                Write-Log "Error: $_"
            }
        } else {
            Write-Log "Appx provisioned package not found: $($remove.Name)"
        }
    }
}

if ($boolOffline) {
    [gc]::Collect()
    Start-Process -FilePath REG.EXE -ArgumentList "UNLOAD HKLM\SWOFF"
    while (Test-Path 'Registry::HKEY_LOCAL_MACHINE\SWOFF') { Start-Sleep -Seconds 3 } # wait for the hive to be unloaded
}

if ($IsAdmin -or $strContext -ne 'USER') {
	# Remove Windows packages
	foreach ($remove in $objWpRemove) {
		if ($boolOffline) {
			$objRemove = Get-WindowsPackage -Path $strOfflinePath | Where { $_.PackageName -like "$($remove.Name)*" }
		} else {
			$objRemove = Get-WindowsPackage -Online | Where { $_.PackageName -like "$($remove.Name)*" }
		}
		if ($objRemove -ne $null) {
			Write-Log "Removing Windows package: $($objRemove.PackageName)"
			try {
				if ($boolOffline) {
					Remove-WindowsPackage -Path $strOfflinePath -PackageName $objRemove.PackageName
				} else {
					Remove-WindowsPackage -PackageName $objRemove.PackageName -Online -NoRestart
				}
				Write-Log "Success"
				Start-Sleep 3
			} catch {
				Write-Log "Error: $_"
			}
		} else {
			Write-Log "Windows package not found: $($remove.Name)"
		}
	}
}

Exit 0